  ' -----------------------------------------------------------
  ' RA02 LoRa module checkout prog.bas
  ' A program for Micromite to communicate with the RA-02 LoRa module
  ' (based on the Semtech SX1278 LoRa chip) via the SPI interface,
  ' for checking its capabilities.
  '
  ' Written by Jim Rowe for Silicon Chip.
  ' Last revision 20/02/2019 at 11:20 am
  ' Updated by Tim Blythman 10/4/2019 to use the same register calls as the Arduino version
  '
  ' NOTE that this program uses an additional Micromite SPI port
  ' to communicate with the RA-02 module, to avoid conflict
  ' with the LCD touchscreen. The added SPI port uses these pins:
  '
  ' SPI Function              Micromite pin
  ' MOSI (SPI out)            Pin 21
  ' MISO (SPI in)             Pin 22
  ' SCK (SPI clock)           Pin 24
  ' NSS/SS-bar (slave select) Pin 26
  '
  ' The embedded C function code needed to support this additional
  ' SPI port can be found at the end of the program.
  '
  ' Two further pins of the Micromite are used to control the RA-02:
  ' Module Reset (RESET).......... connected to Micromite pin 10
  ' Module Dig IO 0 (DIO0) ....... connected to Micromite pin 9
  '
  ' NOTE also that the SPI interface of the SX1278 chip has the
  ' following requirements:
  ' 1. The SS-bar line must be LOW for all command & data transfers
  '    on both the MOSI and MISO lines. This must be controlled by
  '    the program.
  ' 2. Command and data bytes are clocked into the SX1278 via the MOSI
  '    line on the RISING edges of the SCK pulses, which also applies
  '    when data is clocked out of the SX1278 on the MISO line.
  ' 3. Command bytes for data to be written to a register of the
  '    SX1278 must have an MS bit of 1, while those for reading data
  '    from a register must have an MS bit of 0.
  ' 4. All bytes of SPI data sent via the MOSI and MISO lines to or from
  '    the SX1278 should be & are sent MSbit first, LSbit last.
  ' 5. For correct operation, the SPI MODE should be set for 00
  '    -- i.e., CPOL = 0 and CPHA = 0.
  ' 6. We are using MMBasic's Speed 0 for the SPI transactions, which
  '    corresponds to a bit rate/SCK frequency of 1MHz when the Micromite
  '    CPU clock is 40MHz.
  
  ' ---------------------------------------------------------------------------
  OPTION AUTORUN ON
  OPTION EXPLICIT
  OPTION BASE 0
  
  DIM AS INTEGER index, Ctr          ' counters for loops, etc
  DIM AS INTEGER outData, inData     ' variables for SPI transactions
  DIM AS INTEGER TxInLen = 0         ' length of console input string
  DIM AS INTEGER NxtByte = 0         ' ASCII code for next Tx char
  DIM AS INTEGER RxInLen = 0         ' length of received payload packet
  DIM AS INTEGER IntFlag = 0         ' flag to show if screen has been touched
  DIM AS INTEGER SettChg = 0         ' flag to show when settings have changed
  DIM AS INTEGER KeyTime              'count time since last key
  DIM AS STRING TxInput$             ' string for console input
  DIM AS STRING TxBuf$
  DIM AS STRING RxInput$             ' string for LoRa receive input
  DIM AS STRING NxtChar              ' next character from Tx string to send
  DIM AS STRING tmp$                 ' to check inkey$
  
  ' LCD display colours defined
  Const DBlue = RGB(0,0,128)
  CONST Bone = RGB(255,255,192)
  CONST White = RGB(WHITE)
  CONST Black = RGB(BLACK)
  CONST Red = RGB(RED)
  CONST Green = RGB(GREEN)
  
  ShowOpngScrn
  KeyTime=TIMER
  ' now set up the pins for the embedded C function SPI port
  SETPIN 22, DIN            ' make pin 22 a dig input (for MISO)
  PIN(21)= 0:SETPIN 21,DOUT ' set pin 21 low & make it a dig output (MOSI)
  PIN(24)= 0:SETPIN 24,DOUT ' set pin 24 low & make it a dig output (SCK)
  PIN(26)= 1:SETPIN 26,DOUT ' set pin 26 high & make it a dig output (SS-bar)
  SETPIN 9, DIN             ' make pin 9 a dig input (for RA-02 DIO0)
  PIN(10)= 1:SETPIN 10,DOUT ' set pin 10 high & makea dig output (RA-02 RESET)
  
  'SETPIN 15, INTL, TchInt ' call TchInt to set flag when screen touched
  
  PIN(10) = 0     ' drop pin 10 to reset module
  PAUSE 10        ' then wait for 10ms for it to respond
  PIN(10) = 1     ' then raise pin 10 again
  
  PRINT "RA02 LoRa module checkout program"
  PRINT "Tx & Rx frequency set to 434MHz"
  PRINT "Spreading Factor set to 8"
  PRINT "Transmit power set to +14dBm (25mW)"

  'register setup from Arduino code
  outData = &H4200:Send16
  Print "Firmware Version:"; inData AND 255
  outData = &H8180:Send16   'LoRa mode/sleep
  'Set Freq to 434MHz
  outData = &H866C:Send16   'bits 23:16 of RF
  outData = &H8780:Send16   'bits 15: 8 of RF
  outData = &H8800:Send16   'bits 7 : 0 of RF 0x6c8000 = 434 MHz (see datasheet for calcs)
  outData = &H8E00:Send16   'TX fifo address
  outData = &H8F00:Send16   'RX fifo address
  outData = &H8C23:Send16   'LNA, max gain, boost on
  outData = &HA604:Send16   'Modem config3, AGC on
  'outData = &H898F:Send16   'Power Config (see below)
  outData = &H8181:Send16   'LoRa/standby
  'Spreading Factor=8
  outData = &HB1C3:Send16   'Set detection optimization (depends on SF)
  outData = &HB70A:Send16   'Detect Threshold (depends on SF)
  outData = &H9E80:Send16   'SF=8
  outData = &HA604:Send16   'Modem config3, AGC on
  outData = &H898C:Send16   'TX power= +14dBm (25mW)
  outData = &H9D72:Send16   'BW=125kHz, coding=4/5, explicit header
  
  DO
    'check if anything in RX buffer and receive
    outData = &H1200:Send16     'get IRQ flags
    'Print inData AND &H70
    if(inData AND &H70) = &H50 THEN 'RX has occurred
      outData = &H1300:Send16   'get number of RX bytes
      RxInLen=(inData AND 255)
      outData = &H1000:Send16   'get RX buffer pointer
      outData = &H8D00 OR (inData AND 255)  'set FIFO pointer
      Send16
      outData = &H8181:Send16   'set standby mode while processing
      RxInput$=""
      FOR index = 1 to RxInLen
        outData = &H0000:Send16   'get RX FIFO data
        RxInput$=RxInput$ + CHR$(inData AND 255)
      NEXT index
      PRINT "RX Message:";RxInput$
      outData = &H1A00:Send16
      PRINT "RSSI:";(inData AND 255)-157
      outData = &H1900:Send16
      PRINT "SNR:";
      if (inData AND 255)>127 THEN PRINT ((inData AND 255)-256)/4 ELSE PRINT (inData AND 255)/4'print two's complement /4
      outData = &H9250:Send16   'clear IRQ flag
    ENDIF
    outData = &H9220:Send16     'clear CRC error flag
    outData = &H8186:Send16     'receive single mode
    
    'check if TX needs to occur and transmit
    tmp$=inkey$
    if tmp$<>"" THEN KeyTime=TIMER
    TxBuf$=TxBuf$+tmp$          'do stateless keyboard scan
    IF TxBuf$ <> "" and ((TIMER-KeyTime)>100) THEN       ' if there is an input string and no more chars have appeared for a while
      TxInput$=TxBuf$
      PRINT "TX Message:";TxInput$
      TxInLen = LEN(TxInput$)   ' find length of string to be sent
      outData = &H8181:Send16   'LoRa, standby
      outData = &H9D72:Send16   'BW=125kHz, coding=4/5, explicit header
      outData = &H8D00:Send16   'set FIFO pointer to 0
      outData = &HA200 :Send16  'payload length to zero
      FOR index =1 to TxInLen
        outData = &H8000 OR (ASC(MID$(TxInput$,index,1)) AND 255)
        Send16
      NEXT index
      outData = &HA200 OR (TxInLen AND 255):Send16  'payload length
      outData = &H8183:Send16 'TX mode
      inData=0
      DO WHILE ((inData AND &H08)=0)  'wait for TX to complete
        outData = &H1200
        Send16
      LOOP        
      print "Sent"
      outData = &H9208:Send16         'clear TX done flag
      TxBuf$="" 'clear input
    ENDIF
    outData = &H8186:Send16   'receive single mode
  LOOP
END
  
  ' -----------------------------------------------------------------
  ' subroutine to show the opening LCD screen
SUB ShowOpngScrn 
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), DBlue
  TEXT MM.HRes/2, MM.VRes/8, "SILICON CHIP", CM, 1, 3, Red, DBlue
  TEXT MM.HRes/2, MM.VRes/4, "RA-02 LoRa Module", CM, 1, 2, White, DBlue
  TEXT MM.HRes/2, MM.VRes*3/8, "Checkout Program", CM, 1, 2, White, DBlue
  TEXT MM.HRes/2, MM.VRes*4/8, "Please use console", CM, 1, 2, White, DBlue
  
END SUB
  
  SUB Send16
  PIN(26) = 0              ' first bring CSN low to start transaction
  inData = SPIPort(22,21,24,outData,0,0,16) ' then send the 16-bit data
  PIN(26) = 1              ' then bring CSN high to end transaction
  'PRINT "&H";HEX$(outData)"=>&H";HEX$(inData);"<";PIN(9);">"
END SUB                    ' and return
  ' -----------------------------------------------------------------
  ' SPI Port function (Revised by Geoff Graham, 4/8/2017)
CFunction SPIPort(integer, integer, integer, integer) integer
  00000008
  40024800 00442021 40024800 0044102B 1440FFFD 00000000 03E00008 00000000
  27BDFFB0 AFBF004C AFBE0048 AFB70044 AFB60040 AFB5003C AFB40038 AFB30034
  AFB20030 AFB1002C AFB00028 00808821 00A09021 00C0A021 00E0A821 10800005
  8FB30068 10A00004 3C029D00 14C00008 3C109D00 3C029D00 8C420010 00002021
  24050002 0040F809 00003021 3C109D00 8E020010 8E240000 24050002 0040F809
  00003021 8E020024 8E240000 0040F809 00002821 AFA20018 8E020028 0040F809
  8E240000 AFA2001C 8E020024 8E440000 0040F809 24050006 AFA20020 8E020024
  8E440000 0040F809 24050005 AFA20024 8E020028 0040F809 8E440000 24170001
  0057B804 8FA20064 10400008 3C109D00 8C420000 50400006 8E020024 24030003
  5443000D 8E020024 3C109D00 8E020024 8E840000 0040F809 24050005 0040B021
  8E020024 8E840000 0040F809 24050006 1000000A 0040F021 8E840000 0040F809
  24050006 0040B021 8E020024 8E840000 0040F809 24050005 0040F021 3C029D00
  8C420028 0040F809 8E840000 24140001 0054A004 12600002 24110008 8E710000
  2631FFFF 32220020 24030001 02238804 02208021 0002800A 0002880B 2402FFFF
  2403FFFF AFA20010 12A00005 AFA30014 8EA20000 8EA30004 AFA20010 AFA30014
  8FA30060 10600002 0000A821 8C750000 02301025 00009021 10400039 00009821
  8FA40010 02241824 8FA60014 02061024 00621025 10400004 8FA30024 8FA20020
  10000002 AC570000 AC770000 AED40000 00000000 00000000 00000000 00000000
  00000000 00000000 52A00013 00121FC2 02A02021 0411FF6A 00000000 00121FC2
  00131040 00122840 8FA60018 8CC40000 8FA6001C 00C42006 30840001 00A49025
  00629825 AFD40000 0411FF5D 02A02021 1000000C 00101FC0 00131040 00122840
  8FA60018 8CC40000 8FA6001C 00C42006 30840001 00A49025 00629825 AFD40000
  00101FC0 00111042 00621025 00101842 00408821 00431025 1440FFC9 00608021
  00000000 00000000 00000000 00000000 8FA20064 10400006 00000000 8C420000
  10400003 24030002 14430003 02401021 AED40000 02401021 02601821 8FBF004C
  8FBE0048 8FB70044 8FB60040 8FB5003C 8FB40038 8FB30034 8FB20030 8FB1002C
  8FB00028 03E00008 27BD0050
End CFunction

